/**
 *   EasyFS: FileSystem based on HTTP (client for a web server)
 *
 *   Copyright (C) 2008
 *
 *   This program can be distributed under the terms of the GNU LGPL.
 *   
 */

package easyFS;


import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.sun.security.auth.module.UnixSystem;

import cache.Cache;

import constantes.Codes;
import constantes.Type;

import dialogue.Appel;
import donnees.Bdd;
import donnees.BddContent;
import donnees.BddReponse;
import donnees.Fichier;
import donnees.Quota;
import donnees.Repertoire;


import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;

import java.net.InetSocketAddress;
import java.net.Proxy;
import java.nio.ByteBuffer;
import java.nio.BufferOverflowException;
import java.nio.CharBuffer;
import java.util.*;



import fuse.*;
import gnu.getopt.Getopt;
import gnu.getopt.LongOpt;
import donnees.R;


public class EasyFS implements Filesystem3, XattrSupport
{
   // Pour la gestion des logs
   private static final Log log = LogFactory.getLog(EasyFS.class);
   // Pour indiquer le nom du programme
   public static final String PROGRAM_NAME = System.getProperty("program.name", "easyfs");

   // OutputStream pour l'ecriture dans un fichier
   private ByteArrayOutputStream out;

   // Gestion des stats sur le systeme de fichier (le serveur doit normalement les renvoyer)
   private static final int BLOCK_SIZE = 512;
   private static final int NAME_LENGTH = 1024;
   
   // Base de donnees pour le stockage des ids, chemin et noms de fichiers ou repertoires
   private static Bdd bdd = new Bdd();
   
   // Le cache
   private Cache cache;
   
   // La classe appel pour envoyer les requetes au serveur
   private Appel ap;

   // Classe parente pour fichier, lien et repertoire
   private static class N
   {
      static int nfiles = 0;

      String name;
      int mode;
      Map<String,byte[]> xattrs = new HashMap<String,byte[]>();

      N(String name, int mode, String ... xattrs)
      {
         this.name = name;
         this.mode = mode;

         for (int i = 0; i < xattrs.length - 1; i += 2)
            this.xattrs.put(xattrs[i], xattrs[i + 1].getBytes());

         nfiles++;
      }

      public String toString()
      {
         String cn = getClass().getName();
         return cn.substring(cn.indexOf("$")) + "[ name=" + name + ", mode=" + Integer.toOctalString(mode) + "(OCT) ]";
      }
   }

   // Gestion d'un repertoire
   private static class D extends N
   {
      Map<String,N> files = new LinkedHashMap<String,N>();

      D(String name, int mode, String ... xattrs)
      {
         super(name, mode, xattrs);
      }

      void add(N n)
      {
         files.put(n.name, n);
      }

      public String toString()
      {
         return super.toString() + " with " + files.size() + " files";
      }
   }

   // Gestion d'un fichier
   private static class F extends N
   {
      byte[] content;

      F(String name, int mode, byte[] content, String ... xattrs)
      {
         super(name, mode, xattrs);

         this.content = content;
      }
   }

   // Gestion d'un lien
   private static class L extends N
   {
      String link;

      L(String name, int mode, String link, String ... xattrs)
      {
         super(name, mode, xattrs);

         this.link = link;
      }
   }

   // Classe Filhandler associe a la classe parente n
   private static class FH
   {
      N n;

      FH(N n)
      {
         this.n = n;
         log.debug("  " + this + " created");
      }

      void release()
      {
         log.debug("  " + this + " released");
      }

      protected void finalize()
      {
         log.debug("  " + this + " finalized");
      }

      public String toString()
      {
         return "FH[" + n + ", hashCode=" + hashCode() + "]";
      }
   }


   // Rpertoire root
   private D root;
   // Parametres pour l'authentification



   // Pour retrouver un noeud
   private N lookup(String path)
   {
      if (path.equals("/"))
      {
            return root;
      }
      
      File f = new File(path);
      N parent = lookup(f.getParent());
      N node = (parent instanceof D)? ((D) parent).files.get(f.getName()) : null;

      if (log.isDebugEnabled())
         log.debug("  lookup(\"" + path + "\") returning: " + node);

      return node;
   }
   
   // Creation des fichiers
   public void CreerFics(String arb, D root, List<Fichier> listefic) {

	   for (Fichier fic : listefic) 
	   {
		   try {
			   String droits_fuse = "0"+fic.droits;
			   Integer droits = null;
			   droits = Integer.parseInt(droits_fuse,8);
			   
	
			   if (this.cache.getMime(fic.id_fic).contains("unknown"))
				   root.add(new F(fic.nom, droits, this.cache.getContent(fic.id_fic,fic.taille), "mimetype", "text/plain", "description", "a file", "ctime", fic.date_creation, "utime", fic.derniere_modif, "size",fic.taille+""));
			   else
				   root.add(new F(fic.nom, droits, this.cache.getContent(fic.id_fic,fic.taille), "mimetype", this.cache.getMime(fic.id_fic), "description", "a file","ctime", fic.date_creation, "utime", fic.derniere_modif,"size",fic.taille+""));
		} catch (IOException e) {
			log.error("Creation fichier : "+fic.nom);
			//e.printStackTrace();
		}
		   // Ajout dans la bdd
		   bdd.add(new BddContent(fic.id_fic,arb+"/"+fic.nom,fic.nom,constantes.Type.FICHIER, fic.derniere_modif));
	   }
   }
   
   public void CreerReps(String arb, D root, List<Repertoire> listerep) {
	   D tmprep;
	   for (Repertoire rep : listerep) {
		   String droits_fuse = "0"+rep.droits;
		   Integer droits = null;
		   droits = Integer.parseInt(droits_fuse,8);
		   tmprep = new D (rep.nom, droits, "description", "a subdirectory", "utime", rep.derniereModif);
		   root.add(tmprep);
		   // Ajout dans la bdd
		   bdd.add(new BddContent(rep.idRep,arb+"/"+rep.nom,rep.nom,constantes.Type.DOSSIER, rep.derniereModif));
		   CreerFics(arb + "/"+rep.nom, tmprep, rep.listeFichier);
		   if (rep.listeRep!=null)
		   CreerReps(arb + "/"+ rep.nom, tmprep, rep.listeRep);		   
	   }
   }

   // Thread pour le rafraichissement automatique de l'arborescence toutes les 30 s
   class MAJ extends Thread {
	   private boolean threadDone = false;
	   private int sec;
	   
	   MAJ(int s) {
	      this.sec = s*1000;
	   }

	    public void done() {
	        threadDone = true;
	    }

	   public void run() {
	      try {
	         while (!threadDone) {
	            refresh();
	            sleep(sec);
	         }
	      } catch (Exception e) {
	        }   
	   }
	}

   // Constructeur
   public EasyFS(String login, String password, String server, Proxy proxy) throws IOException
   {

	   this.ap = new Appel(proxy,login,password,server);
	   this.cache = new Cache(this.ap, ".cache");
	   Object retour = ap.authentification();
	   if (retour instanceof Repertoire)
	   {
		   root = new D("", 0755, "description", "ROOT directory");
		   String droits_fuse = "0"+((Repertoire)retour).droits;
		   Integer droits = null;
		   droits = Integer.parseInt(droits_fuse,8);
		   D repgrp = new D(((Repertoire)retour).nom, droits, "description", "Group directory", "utime", ((Repertoire)retour).derniereModif);
		   root.add(repgrp);
		   // On cree les fichiers du repertoire de groupe
		   CreerFics("/"+((Repertoire)retour).nom, repgrp, ((Repertoire)retour).listeFichier);
		   bdd.add(new BddContent(((Repertoire)retour).idRep,"/"+((Repertoire)retour).nom,((Repertoire)retour).nom,constantes.Type.DOSSIER, ((Repertoire)retour).derniereModif));
		   CreerReps("/"+((Repertoire)retour).nom, repgrp, ((Repertoire)retour).listeRep);
		   // On fait un refresh toutes les 30 secondes
		   MAJ majarbo = new MAJ(30);
		   majarbo.start();
		   log.info("created");
		   
	   }      
	   else { 
		   log.error("ERREUR DE CONNEXION!!!!!!!!!!!");
		   System.exit(-1);
	   }
	   
   }
   
   // Rafraichissement de l'arborescence
   public void refresh() throws IOException {
	   if(ap.besoinMAJ()){
		   // On vide les informations ou on les reinitialise
		   Object retour = ap.demanderArb();
		   cache = new Cache(ap, ".cache");
		   root.files.clear();
		   root = null;
		   // Vidage de la bdd
		   bdd.truncate();
		   if (retour instanceof Repertoire)
		   {
			   int time = (int) (System.currentTimeMillis() / 1000L);
			   root = new D("", 0755, "description", "ROOT directory");
			   String droits_fuse = "0"+((Repertoire)retour).droits;
			   Integer droits = null;
			   droits = Integer.parseInt(droits_fuse,8);
			   D repgrp = new D(((Repertoire)retour).nom, droits, "description", "Group directory", "utime", ((Repertoire)retour).derniereModif);
			   root.add(repgrp);
			   // On cree les fichiers du r???pertoire de groupe
			   CreerFics("/"+((Repertoire)retour).nom, repgrp, ((Repertoire)retour).listeFichier);
			   bdd.add(new BddContent(((Repertoire)retour).idRep,"/"+((Repertoire)retour).nom,((Repertoire)retour).nom,constantes.Type.DOSSIER, ((Repertoire)retour).derniereModif));
			   CreerReps("/"+((Repertoire)retour).nom, repgrp, ((Repertoire)retour).listeRep);
			   log.info("created");
		   }
	   } 
	   else
		   log.info("Pas besoin de maj");
   }
   
   
    // Pour changer les permissions
   public int chmod(String path, int mode) throws FuseException
   {
	  
	  // Pour le retour
	  int renvoi = 0;
	  String perm = Integer.toOctalString(mode);
	  String mod = perm.substring(perm.length()-3,perm.length());
	 // System.out.println("Permissions : "+mod);
	  Integer droit = Integer.parseInt(mod);
	  
	  try {
			// On recupere l'id du rep parent a partir du chemin
			BddReponse id = bdd.searchIdfrompath(path);
			// On le modifie sur le serveur
			 switch (id.type()) {
		      case Type.FICHIER :
		    	  renvoi = ap.chmodF(id.id(), droit);
		    	  break;
		      case Type.DOSSIER : 
		    	  renvoi = ap.chmodD(id.id(), droit);
		    	  break;
		      }
			//refresh();
	  } catch (IOException e) {
		  log.error("Chmod sur : "+path);
	  	}
	  if(renvoi == Codes.OK){
      N n = lookup(path);
         n.mode = (n.mode & FuseStatConstants.TYPE_MASK) | (mode & FuseStatConstants.MODE_MASK);
         System.out.println("NMode!!!!!!!! :" +n.mode);
      }

     return renvoi;
   }

   
   public int chown(String path, int uid, int gid) throws FuseException
   {
      return 0;
   }

   // Recuperer les attributs d'un fichier ou repertoire
   public int getattr(String path, FuseGetattrSetter getattrSetter) throws FuseException
   {
      N n = lookup(path);

      UnixSystem s = new UnixSystem();
      
      
      
      if (n == null)
          throw new FuseException("No Such Entry").initErrno(FuseException.ENOENT);

      int time = (int) (System.currentTimeMillis() / 1000L);
      

      if (n instanceof D)
      {
         D d = (D) n;

         int utime;
         try {
          utime = Integer.parseInt(new String(d.xattrs.get("utime"))); 
         } catch (Exception e) {utime = time;}
         
         getattrSetter.set(
            d.hashCode(),
            FuseFtypeConstants.TYPE_DIR | d.mode,
            1,
            (int) s.getUid(), (int) s.getGid(),
            0,
            d.files.size() * NAME_LENGTH,
            (d.files.size() * NAME_LENGTH + BLOCK_SIZE - 1) / BLOCK_SIZE,
            time, utime, utime
         );

         return 0;
      }
      else if (n instanceof F)
      {
         F f = (F) n;
         int ctime = Integer.parseInt(new String(f.xattrs.get("ctime")));
         int utime = Integer.parseInt(new String(f.xattrs.get("utime"))); 
         int size = Integer.parseInt(new String(f.xattrs.get("size")));
         
         getattrSetter.set(
            f.hashCode(),
            FuseFtypeConstants.TYPE_FILE | f.mode,
            1,
            (int) s.getUid(), (int) s.getGid(),
            0,
            size,//f.content.length,
            (size + BLOCK_SIZE - 1) / BLOCK_SIZE,
           time, utime, ctime
         );

         return 0;
      }
      else if (n instanceof L)
      {
         L l = (L) n;
         getattrSetter.set(
            l.hashCode(),
            FuseFtypeConstants.TYPE_SYMLINK | l.mode,
            1,
            (int) s.getUid(), (int) s.getGid(),
            0,
            l.link.length(),
            (l.link.length() + BLOCK_SIZE - 1) / BLOCK_SIZE,
            time, time, time
         );

         return 0;
      }

      return Errno.ENOENT;
   }

   // Gestion des repertoires
   public int getdir(String path, FuseDirFiller filler) throws FuseException
   {
      N n = lookup(path);

      if (n instanceof D)
      {
         for (N child : ((D) n).files.values())
         {
            int ftype = (child instanceof D)
                        ? FuseFtypeConstants.TYPE_DIR
                        : ((child instanceof F)
                           ? FuseFtypeConstants.TYPE_FILE
                           : ((child instanceof L)
                           ? FuseFtypeConstants.TYPE_SYMLINK
                           : 0));
            if (ftype > 0)
               filler.add(
                  child.name,
                  child.hashCode(),
                  ftype | child.mode
               );
         }

         return 0;
      }

      return Errno.ENOTDIR;
   }

   // Creation d'un lien (non implment)
   public int link(String from, String to) throws FuseException
   {
      return Errno.EROFS;
   }
   
   // La classe trs pratique pour la rcupration du chemin et du nom a partir
   // d'un chemin complet
   protected class Spath {
	   String chemin="";
	   String nom="";
	   Spath (String chemin, String nom) {
		   this.chemin = chemin;
		   this.nom = nom;
	   }
	   Spath (String cpath) {
		   if( cpath.indexOf( "/" ) != -1 )
			  {
				  String split[] = cpath.split( "/" );
				  this.nom = split[split.length-1];
				  
				  for( int i = 0; i < split.length-1; i++ )
				  {
						  this.chemin += split[i] + "/";
				  }
				  // On enleve le dernier / (pour etre coherent avec Fuse)
				  chemin = chemin.substring(0, chemin.length()-1);
			  }
	   }
	   
	   public String toString() {
			return chemin.toString() +"/"+ nom.toString();
		}
   }

   // Creation d'un repertoire
   public int mkdir(String path, int mode) throws FuseException
   {
	  // On recupere le nom et le chemin
	  Spath p = new Spath(path);


		  try {
			// On recupere l'id du rep parent a partir du chemin
			BddReponse id = bdd.searchIdfrompath(p.chemin);
			// On le cree sur le serveur
			R retour = ap.creerRepertoire(p.nom, id.id(), 640);
			if(retour.code == Codes.OK){
				N n = lookup(p.chemin);		
				Integer time = (int) (System.currentTimeMillis() / 1000L);
				((D) n).add( new D (p.nom, 0640, "description", "a subdirectory", "utime", time.toString()));				
				bdd.add(new BddContent(retour.param,path,p.nom,constantes.Type.DOSSIER, retour.s));
				return retour.code;
				}
		} catch (IOException e) {
			log.error("Erreur creation repertoire :"+p.nom);
			//e.printStackTrace();
		}
		return 0;
   }

   // Creation d'un fichier
   public int mknod(String path, int mode, int rdev) throws FuseException
   {
	    
	    Spath nouv = new Spath(path);
	    if(nouv.chemin.lastIndexOf("~") != (nouv.chemin.length()) ){ //on ne s'occupe pas des fichiers de sauvegarde !
		    try {
				R retour = ap.touchFile(nouv.nom,(bdd.searchIdfrompath(nouv.chemin)).id(),640);
				if(retour.code == Codes.OK){
				N n = lookup(nouv.chemin);	
				int time = (int) (System.currentTimeMillis() / 1000L);
				((D) n).add(new F(nouv.nom, 0640, new byte[0], "mimetype", "application/octet-stream", "description", "a file", "ctime", ""+time, "utime", ""+time, "size", "0"));				
				bdd.add(new BddContent(retour.param,path,nouv.nom,constantes.Type.FICHIER, retour.s));	
				}
				
				return retour.code;
		    } 
			 catch (Exception e) {
				
				e.printStackTrace();
				 log.error("Creation fichier : "+nouv);
			}		
	    }
	    return 0;
   }

   // Renommage ou dplacement
   public int rename(String from, String to) throws FuseException
   {
	   BddReponse bdrep = bdd.searchIdfrompath(from);
	   Spath bef = new Spath(from);
	   Spath aft = new Spath(to);
	   
		  Object retour = null;
	      switch (bdrep.type()) {
	      case Type.FICHIER :
	    	  try {
	    		    // Si le chemin est pareil mais le nom est different, on renomme
	    		    if ((bef.chemin.equals(aft.chemin)) && (!bef.nom.equals(aft.nom)))
					retour = ap.renommerFichier(bdrep.id(), aft.nom);
	    		    else { 
	    		    	// On deplace (chemin different mais le nom est pareil)
	    		    	// Prendre l'id du nouveau repertoire
	    		    	BddReponse nrep = bdd.searchIdfrompath(aft.chemin);
	    		    	if (nrep!=null)
	    		    	retour = ap.deplacerFichier(bdrep.id(), nrep.id());
	    		    }
				} catch (IOException e1) {
					log.error("Renommage ou deplacement de : "+from+" vers :"+to);
				}
	    	  break;
	      case Type.DOSSIER : 
	    	  try {
	    		  // Si le chemin est pareil mais le nom est different, on renomme
	    		    if ((bef.chemin.equals(aft.chemin)) && (!bef.nom.equals(aft.nom)))
					retour = ap.renommerRepertoire(bdrep.id(), aft.nom);
	    		    else {
                        // On d???place (chemin different mais le nom est pareil)
	    		    	BddReponse nrep = bdd.searchIdfrompath(aft.chemin);
	    		    	if (nrep!=null)
	    		    	retour = ap.deplacerRepertoire(bdrep.id(), nrep.id());
	    		    }
				} catch (IOException e1) {
					log.error("Renommage ou deplacement de : "+from+" vers :"+to);
				}
	    	  break;
	      }
		  try {
			// On rafraichit l'arborescence
			this.refresh();
		} catch (IOException e) {
			log.error("Refresh Renommage ou deplacement de : "+from+" vers :"+to);
		}
		
	    return (Integer) retour;
   }

   // Suppression d'un repertoire
   public int rmdir(String path) throws FuseException
   {
	   Spath nouv = new Spath(path);
	   int retour = 0;
	   BddReponse bdrep = bdd.searchIdfrompath(path);
	    	  try {
				retour = ap.supprimerRepertoire(bdrep.id());
				if(retour == Codes.OK){
					bdd.del(bdrep.id());
					D d =(D) lookup(nouv.chemin);
					d.files.remove(nouv.nom);
				}
			} catch (IOException e) {
				log.error("Rmdir : "+path);
			}
	  return retour;
   }

   // Pour les statistiques du systeme de fichier
   public int statfs(FuseStatfsSetter statfsSetter) throws FuseException
   {
	   Object stat = null;
	   Quota q = null;
	try {
		stat = ap.getQuota();
	} catch (IOException e) {
		// TODO Auto-generated catch block
		e.printStackTrace();
	}
	   if (stat instanceof Quota)
		   q = (Quota) stat;
		 
	   //Fetch the filesystem information.
      statfsSetter.set(
         BLOCK_SIZE,
         q.taille_max/BLOCK_SIZE, //max
         (q.taille_max - q.taille)/BLOCK_SIZE, //libre
         -1,
         N.nfiles,
         N.nfiles,
         NAME_LENGTH
      );
      
      /*$namelen, $files, $files_free, $blocks, $blocks_avail, $blocksize

      or

      -ENOANO(), $namelen, $files, $files_free, $blocks, $blocks_avail, $blocksize*/

      return 0;
   }

   public int symlink(String from, String to) throws FuseException
   {
      return Errno.EROFS;
   }

   // Changer la taille d'un fichier
   public int truncate(String path, long size) throws FuseException
   {
	//   Spath p = new Spath(path);
	  // Pour le retour
	 // Object renvoi = null;
	  System.out.println("truncate "+path+" "+size);
	/*  try {
			// On recupere l'id du rep parent a partir du chemin
			BddReponse id = bdd.searchIdfrompath(p.chemin);
			// On le modifie sur le serveur
			//renvoi = ap.changesize(p.nom, id.id(), 640);
			refresh();
	  } catch (IOException e) {
		  log.error("Truncate sur : "+p);
	  	}*/
	  return 0;
   }

   // Suppression d'un fichier
   public int unlink(String path) throws FuseException
   {
	   Spath nouv = new Spath(path);
	   BddReponse bdrep = bdd.searchIdfrompath(path);
	    	  try {
				int retour = ap.supprimerFichier(bdrep.id());
				if(retour == Codes.OK){
				bdd.del(bdrep.id());
				D d =(D) lookup(nouv.chemin);
				d.files.remove(nouv.nom);
				}
				
			} catch (IOException e) {
				 log.error("unlink sur : "+bdrep);
			}
	  return 0;
   }

   // Changer la derniere modification d'un fichier
   public int utime(String path, int atime, int mtime) throws FuseException
   {
	   return 0;
   }

   public int readlink(String path, CharBuffer link) throws FuseException
   {
      N n = lookup(path);

      if (n instanceof L)
      {
         link.append(((L) n).link);
         return 0;
      }

      return Errno.ENOENT;
   }

   // Ouverture d'un fichier
   public int open(String path, int flags, FuseOpenSetter openSetter) throws FuseException
   {
      N n = lookup(path);
      if(n!=null){
    
      out = new ByteArrayOutputStream();
      openSetter.setFh(new FH(n));
    	  
      return Codes.OK;
      
      
      }
      else
    	  return Codes.ENOENT;
   }

  
   
   
   public String getMime(String name){
	   String[] ext =  name.split(".");
	   if(ext.length == 0)
		   return null;
	   else
		   if (ext[ext.length].equalsIgnoreCase("jpg") || ext[ext.length].equalsIgnoreCase("jpeg"))
			   return "image/jpeg";
	   
	   	   if(ext[ext.length].equalsIgnoreCase("png"))
	   		   return "image/png";
	   	   
	   	   if(ext[ext.length].equalsIgnoreCase("gif"))
	   		   return "image/gif";
	   	   
	   	   if(ext[ext.length].equalsIgnoreCase("txt") || ext[ext.length].equalsIgnoreCase("c") || ext[ext.length].equalsIgnoreCase("php") || ext[ext.length].equalsIgnoreCase("html") || ext[ext.length].equalsIgnoreCase("js"))
	   		   return "text/plain";
	   
	   return null;
   }
   // Pour l'ecriture dans un fichier (modification)
   public int write(String path, Object fh, boolean isWritepage, ByteBuffer buf, long offset) throws FuseException
   { 

	    // On remplit le output stream
	    byte[] buffer = new byte[buf.capacity()];
		buf.get(buffer);
		if (fh instanceof FH)
	      {
	         F f1 = (F) ((FH) fh).n;
	         System.out.println("MIME : " +new String(f1.xattrs.get("mimetype")));	         
	      }
		
		try {
			out.write(buffer);
		} catch (IOException e) {
			log.error("write sur : "+path);
		}

		return 0;
   }
   
   private int getsize(String s){
	   F n = (F) lookup(s);
	   return Integer.parseInt(new String(n.xattrs.get("size")));
   }

   // Lecture d'un fichier
   public int read(String path, Object fh, ByteBuffer buf, long offset) throws FuseException
   {
      if (fh instanceof FH)
      {
         F f = (F) ((FH) fh).n;
         BddReponse reponse = bdd.searchIdfrompath(path);

         if(!cache.fichierDsCache(reponse.id()) ||  !cache.fichierAjour(reponse.id())){
				try {
					cache.miseEnCache(reponse.id());
				} catch (IOException e) {
					log.error("mise en cache dans open sur : "+path);
				}
         }
         try {
			f.content = cache.getContent(reponse.id(),getsize(path));			
		} catch (IOException e) {
			// TODO Auto-generated catch block
			log.info("read sur : "+path);
		}
         buf.put(f.content, (int) offset, Math.min(buf.remaining(), f.content.length - (int)offset));
         return 0;
      }
         
      return Errno.EAGAIN;
   }

   // Fermeture fichier
   public int flush(String path, Object fh) throws FuseException
   {
	   if (fh instanceof FH ){
		   if(out.size() != 0){
		  BddReponse reponse = bdd.searchIdfrompath(path);
  
		  F f = (F) ((FH) fh).n;
		  String mime = new String(f.xattrs.get("mimetype"));
		  String mimeext = getMime(f.name);
		  
		    if(mimeext == null && mime.equals("fichier inexistant"))
		    	mime = "application/octet-stream";
		    else if(mimeext != null)
		    	mime = mimeext;
		    
	      
		    try{
		    	int r =  (Integer) ap.upContenu(reponse.id(), out,mime);
		    	if(r == Codes.OK){
		    		F n = (F) lookup(path);
		    		n.xattrs.put("size",(new String(out.size()+"").getBytes()));
		    	}
		    	return r;
		    } catch (IOException e) {
		    	log.error("write sur : "+path);
		    }
		
		//On met le fichier a jour dans le cache
		    try {
		    	cache.miseEnCache(reponse.id());
		    } catch (IOException e) {
		    	log.info("Mise en cache write sur : "+path);
		    }
		   }
		   return 0;
	   }
      return Errno.EBADF;
   }

   // Rafraichissement (synchronisation des donnees)
   public int fsync(String path, Object fh, boolean isDatasync) throws FuseException
   {
	  try {
		refresh();
	} catch (IOException e) {
		// TODO Auto-generated catch block
		e.printStackTrace();
	}
      if (fh instanceof FH)
         return 0;

      return Errno.EBADF;
   }

   // Quand le fichier est totalement ferme
   public int release(String path, Object fh, int flags) throws FuseException
   {
      if (fh instanceof FH)
      {
         ((FH) fh).release();
         // A process has closed the file
         System.runFinalization();
         return 0;
      }

      return Errno.EBADF;
   }

   //
   // XattrSupport implementation

   /**
    * This method will be called to get the value of the extended attribute
    *
    * @param path the path to file or directory containing extended attribute
    * @param name the name of the extended attribute
    * @param dst  a ByteBuffer that should be filled with the value of the extended attribute
    * @return 0 if Ok or errno when error
    * @throws fuse.FuseException an alternative to returning errno is to throw this exception with errno initialized
    * @throws java.nio.BufferOverflowException
    *                            should be thrown to indicate that the given <code>dst</code> ByteBuffer
    *                            is not large enough to hold the attribute's value. After that <code>getxattr()</code> method will
    *                            be called again with a larger buffer.
    */
   public int getxattr(String path, String name, ByteBuffer dst) throws FuseException, BufferOverflowException
   {
      N n = lookup(path);

      if (n == null)
         return Errno.ENOENT;

      byte[] value = n.xattrs.get(name);

      if (value == null)
         return Errno.ENOATTR;

      dst.put(value);

      return 0;
   }

   /**
    * This method can be called to query for the size of the extended attribute
    *
    * @param path       the path to file or directory containing extended attribute
    * @param name       the name of the extended attribute
    * @param sizeSetter a callback interface that should be used to set the attribute's size
    * @return 0 if Ok or errno when error
    * @throws fuse.FuseException an alternative to returning errno is to throw this exception with errno initialized
    */
   public int getxattrsize(String path, String name, FuseSizeSetter sizeSetter) throws FuseException
   {
      N n = lookup(path);

      if (n == null)
         return Errno.ENOENT;

      byte[] value = n.xattrs.get(name);

      if (value == null)
         return Errno.ENOATTR;

      sizeSetter.setSize(value.length);

      return 0;
   }

   /**
    * This method will be called to get the list of extended attribute names
    *
    * @param path   the path to file or directory containing extended attributes
    * @param lister a callback interface that should be used to list the attribute names
    * @return 0 if Ok or errno when error
    * @throws fuse.FuseException an alternative to returning errno is to throw this exception with errno initialized
    */
   public int listxattr(String path, XattrLister lister) throws FuseException
   {
      N n = lookup(path);

      if (n == null)
         return Errno.ENOENT;

      for (String xattrName : n.xattrs.keySet())
         lister.add(xattrName);

      return 0;
   }

   /**
    * This method will be called to remove the extended attribute
    *
    * @param path the path to file or directory containing extended attributes
    * @param name the name of the extended attribute
    * @return 0 if Ok or errno when error
    * @throws fuse.FuseException an alternative to returning errno is to throw this exception with errno initialized
    */
   public int removexattr(String path, String name) throws FuseException
   {
      return Errno.EROFS;
   }

   /**
    * This method will be called to set the value of an extended attribute
    *
    * @param path  the path to file or directory containing extended attributes
    * @param name  the name of the extended attribute
    * @param value the value of the extended attribute
    * @param flags parameter can be used to refine the semantics of the operation.<p>
    *              <code>XATTR_CREATE</code> specifies a pure create, which should fail with <code>Errno.EEXIST</code> if the named attribute exists already.<p>
    *              <code>XATTR_REPLACE</code> specifies a pure replace operation, which should fail with <code>Errno.ENOATTR</code> if the named attribute does not already exist.<p>
    *              By default (no flags), the  extended  attribute  will  be created if need be, or will simply replace the value if the attribute exists.
    * @return 0 if Ok or errno when error
    * @throws fuse.FuseException an alternative to returning errno is to throw this exception with errno initialized
    */
   public int setxattr(String path, String name, ByteBuffer value, int flags) throws FuseException
   {
      return Errno.EROFS;
   }

   // Pour montrer a l'utilisateur comment utiliser easyFS
   protected static void displayUsage()
   {
	   System.out.println("Easyfs : a client to mount a remote disk space with fuse.");
       System.out.println();
       System.out.println("usage: " + PROGRAM_NAME + " [options]");
       System.out.println();
       System.out.println("options:");
       System.out.println("    -h, --help                Show this help message (default)");
       System.out.println("    -d, --directory=<path>    The directory to mount the filesystem");
       System.out.println("    --                        Stop processing options");
       System.out.println("    -s, --server=<url>        Specify the URL of the remote server");
       System.out.println("    -u, --user=<name>         Specify the username for authentication");
       System.out.println("    -p, --password=<name>     Specify the password for authentication");
       System.out.println("    -P, --proxy=<url>         Specify the proxy to use");
       System.out.println("    -c, --proxyport=<int>     Specify the proxy port to use");
       System.out.println("    -V, --version             Show the version and exit");
       System.out.println("    -g, --graphic             Graphic version for the command options");
       System.out.println();
       System.out.println("Use example (text mode) : ");
       System.out.println("./easyfs_mount.sh -u name -p password -d /mnt/directory -shttp://server.com/server.php -P195.83.8.14 -c985");
   }

  
   // Java entry point
   public static void main(String[] args)
   {
	  boolean graphic = false, debug = false;
	  String dir = null, password = null, proxy=null, user = null, server = null;
	  Integer port = null;
	  int code;
        
	  // : requis, :: optionnel, sans rien => aucun argument
      String sopts = "hd:u:p:s::P::c::gVD";
      LongOpt[] lopts =
    	         {
    	            new LongOpt("help", LongOpt.NO_ARGUMENT, null, 'h'),
    	            new LongOpt("dir", LongOpt.REQUIRED_ARGUMENT, null, 'd'),
    	            new LongOpt("debug", LongOpt.REQUIRED_ARGUMENT, null, 'D'),
    	            new LongOpt("password", LongOpt.REQUIRED_ARGUMENT, null, 'p'),
    	            new LongOpt("proxy", LongOpt.OPTIONAL_ARGUMENT, null, 'P'),
    	            new LongOpt("proxyport", LongOpt.OPTIONAL_ARGUMENT, null, 'c'),
    	            new LongOpt("version", LongOpt.NO_ARGUMENT, null, 'V'),
    	            new LongOpt("user", LongOpt.REQUIRED_ARGUMENT, null, 'u'),
    	            new LongOpt("server", LongOpt.OPTIONAL_ARGUMENT, null, 's'),
    	         };
      
      Getopt getopt = new Getopt("easyfs", args, sopts, lopts);
     
	  while ((code = getopt.getopt()) != -1)
	  {
		  switch (code)
	         {
	            case ':':
	            case '?':
	               System.exit(1);
	               break;

	            case 1:
	               // Si argument en trop
	               System.err.println("easyfs" + ": unused non-option argument: " +
	               getopt.getOptarg());
	               break;
	               
	            case 'h':
	            	displayUsage();
	                System.exit(0);
	                break;
	                
	            case 'u':
	            {
		               user = getopt.getOptarg();
		               break;
	            }      
	            case 'c':
	            {
		               port = (new Integer(getopt.getOptarg())).intValue();
		               //SecurityAssociation.setPrincipal(new SimplePrincipal(username));
		               break;
	            }      
	            case 'd':
	            {
	               // directory to mount
	               dir = getopt.getOptarg();
	               break;
	            }
	            case 'D':
	            {
	               // directory to mount
	               debug = true;
	               break;
	            }
	            case 's':
	               server = getopt.getOptarg();
	               break;

	            case 'p':
	               password = getopt.getOptarg();
	               break;
	            case 'P':
		               proxy = getopt.getOptarg();
		               break;
	            case 'V':
	            	Package easyfsPackage = Package.getPackage("client.easyFS");
	            	// Montrer les informations sur la version
	            	System.out.println("EasyFS " + easyfsPackage.getImplementationVersion());
	            	System.out.println();
	            	System.out.println("Distributable under LGPL license.");
	            	System.out.println("See terms of license at gnu.org.");
	            	System.out.println();
	            	System.exit(0);
	            	break;
	            case 'g':
	            	graphic = true;
	            	break;
	            default:
	            	throw new Error("unhandled option code: " + code);
	         }
	  }

	  // Si un des parametres n'est pas renseigne ou mode graphique
	  if ((dir==null || dir.equals("") || password==null || user==null) || graphic || (port==null && proxy!=null) || (proxy==null && port!=null))
	  {
    	     // On essaye de passer en mode graphique
    	  	 try {
    	     EasyFSLogin frame = new EasyFSLogin(dir, password, user, server, proxy, port);
    	     
    	 	 while(frame.send!=true) {};
    	 	 graphic = true;
    	 	 
    	 	 // On recupere les informations
    	 	 if (!frame.user.getText().equals(""))
    	 	 user = frame.user.getText();
    	 	 
    	 	 if (!frame.pass.getText().equals(""))
    	 	 password = frame.pass.getText();
    	 	 
    	 	 if (!frame.mntpoint.getText().equals(""))
    	     dir = frame.mntpoint.getText();
    	 	 
    	 	 if (!frame.serv.getText().equals("defaut"))
    	 		 server = frame.serv.getText();
    	 	 
    	 	 if (!frame.proxy.getText().equals("aucun"))
    	 		 proxy = frame.proxy.getText();
    	 	 
    	 	if (!frame.port.getText().equals("aucun"))
    	 		port = (new Integer(frame.port.getText())).intValue();
    	 	 
    	  	 }
    	  	 catch (Exception e) {
    	  		 //e.printStackTrace();
    	  		System.err.println("No X11 interface defined, impossible to launch the graphic mode!");
    	  		displayUsage();
    	  		System.exit(-1);
    	  	 }
	  }
	  else if ((dir==null || password==null || user==null) && !graphic || (port==null && proxy!=null) || (proxy==null && port!=null)) {
		  displayUsage();
          System.exit(0);
	  }
	  
      String[] fusearg = new String[2]; 
      fusearg[0] = dir;
      
      // -d : debug mode
      // -f : foreground (with log messages)
      // -s : disable multithreaded operation
      if (debug)
      fusearg[1] = "-d";
      else
      fusearg[1] = "-f";
    
      if (server==null || server.equals(""))
    	  server="http://fuse.credi-ong.org/serveur/parse_requetes.php";
      
     
	 
	  log.info("entering");
	  
      try
      {
    	  if (graphic) {
    		  System.out.println(dir);
    		  		Runtime.getRuntime().exec("XandrosFileManager file://"+dir);
	    			}
    	  
    	  
    	  if (proxy==null || proxy.equals("")) {
    	  FuseMount.mount(fusearg, new easyFS.EasyFS(user, password, server, null), log);
          }
    	  else {
        	  Proxy prox = new Proxy(Proxy.Type.HTTP, new InetSocketAddress(proxy, port));
        	  FuseMount.mount(fusearg, new easyFS.EasyFS(user, password, server, prox), log);
    	  } 
    	  
    	 

          
      }
      catch (Exception e)
      {
    	  log.error("Impossible to mount : FuseMount.mount");
      }
      finally
      {
         log.info("exiting");
      }
   }
}